﻿import express from "express";
import { randomUUID } from "node:crypto";
import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
import { StreamableHTTPServerTransport } from "@modelcontextprotocol/sdk/server/streamableHttp.js";
import { SSEServerTransport } from "@modelcontextprotocol/sdk/server/sse.js";
import { isInitializeRequest } from "@modelcontextprotocol/sdk/types.js";
import { z } from "zod";

const app = express();
app.use(express.json({ limit: "10mb" }));

const authToken = process.env.MCP_AUTH_TOKEN || "";
if (authToken) {
  app.use((req, res, next) => {
    const auth = req.headers.authorization || "";
    if (auth !== `Bearer ${authToken}`) {
      res.status(401).json({ error: "Unauthorized" });
      return;
    }
    next();
  });
}

const createServer = () => {
  const server = new McpServer({
    name: "edesk-wp",
    version: "0.1.0",
  });

  server.tool(
    "ping",
    { message: z.string().optional() },
    async ({ message }) => ({
      content: [{ type: "text", text: message ?? "pong" }],
    })
  );

  return server;
};

const streamableSessions = new Map();
const sseSessions = new Map();

app.post("/mcp", async (req, res) => {
  try {
    const sessionIdHeader = req.headers["mcp-session-id"];
    let session = null;

    if (sessionIdHeader && streamableSessions.has(sessionIdHeader)) {
      session = streamableSessions.get(sessionIdHeader);
    } else if (!sessionIdHeader && isInitializeRequest(req.body)) {
      const transport = new StreamableHTTPServerTransport({
        sessionIdGenerator: () => randomUUID(),
      });
      const server = createServer();
      session = { transport, server };
      streamableSessions.set(transport.sessionId, session);
    } else {
      const transport = new StreamableHTTPServerTransport();
      const server = createServer();
      session = { transport, server };
    }

    await session.server.connect(session.transport);
    await session.transport.handleRequest(req, res, req.body);
  } catch (err) {
    console.error("MCP /mcp error:", err);
    if (!res.headersSent) {
      res.status(500).json({ error: "MCP transport error" });
    }
  }
});

app.get("/sse", async (req, res) => {
  try {
    const transport = new SSEServerTransport("/messages", res);
    const server = createServer();
    sseSessions.set(transport.sessionId, { transport, server });
    res.on("close", () => sseSessions.delete(transport.sessionId));
    await server.connect(transport);
  } catch (err) {
    console.error("MCP /sse error:", err);
    if (!res.headersSent) {
      res.status(500).json({ error: "SSE transport error" });
    }
  }
});

app.post("/messages", async (req, res) => {
  const sessionId = req.query.sessionId;
  const session = sseSessions.get(sessionId);
  if (!session) {
    res.status(400).json({ error: "Unknown SSE session" });
    return;
  }
  try {
    await session.transport.handlePostMessage(req, res, req.body);
  } catch (err) {
    console.error("MCP /messages error:", err);
    if (!res.headersSent) {
      res.status(500).json({ error: "SSE message error" });
    }
  }
});

app.get("/mcp", (req, res) => {
  res.status(405).json({ error: "Use POST /mcp (Streamable HTTP) or /sse" });
});

const okBody = Buffer.from("ok");
const sendOk = (res) => {
  res.status(200);
  res.setHeader("Content-Type", "text/html");
  res.setHeader("Content-Length", okBody.length);
  res.end(okBody);
};

app.get("/", (req, res) => sendOk(res));
app.get("/health", (req, res) => sendOk(res));

const port = process.env.PORT || 3000;
app.listen(port, () => {
  console.log(`MCP server listening on :${port}`);
});
